<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Inbox extends Model {
    use HasFactory;

    protected $guarded = ['id'];
    protected $casts = [
        'date' => 'datetime'
    ];

    public function user() {
        return $this->belongsTo(User::class);
    }

    public function conversations() {
        return $this->hasMany(Conversation::class);
    }

    public function getLatestConversationAttribute() {
        return $this->conversations()->latest()->first();
    }

    public static function search($string) {
        return parent::where(function ($query) use ($string) {
            $query->where('ref', 'like', "%$string%")
                ->orWhere('subject', 'like', "%$string%")
                ->orWhere('status', 'like', "%$string%")
                ->orWhereHas('user', function ($q) use ($string) {
                    $q->where('name', 'like', "%$string%");
                });
        });
    }


    public function scopeOfDate(Builder $query, string $fromDate, string $toDate): void {
        if (filled($fromDate) && filled($toDate)) {
            $toDate = Carbon::parse($toDate)->addDay()->format('Y-m-d');
            $fromDate = Carbon::parse($fromDate)->addDay()->format('Y-m-d');
            $query->whereBetween('created_at', [$fromDate, $toDate]);
        }
    }
}
