<?php

namespace App\Http\Controllers\User;

use App\Models\User;
use App\Models\Settings;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller {

    public function index() {
        $settings = Settings::where('id', 1)->first();
        return view("{$settings->theme}.user.profile.details", [
            'title' => 'User Profile',
            'user' => User::find(auth()->id()),
        ]);
    }

    public function edit() {
        $settings = Settings::where('id', 1)->first();
        return view("{$settings->theme}.user.profile.edit", [
            'title' => 'Edit User Profile',
            'user' => User::find(auth()->id()),
        ]);
    }

    public function updatePersonalDetails(Request $request) {
        $personalDetails = $request->validate(
            [
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255',
                'dob' => 'required|date',
                'country' => 'required|string',
            ],
            [],
            [
                'dob' => 'date of birth',
            ]
        );

        $user = User::find(auth()->id());
        $user->update($personalDetails);

        return redirect(route('profile'))->with('success', 'Account updated.');
    }

    public function updatePassword(Request $request) {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required',
        ]);

        $user = User::find(auth()->id());

        if (!Hash::check($request->current_password, $user->password)) {
            return back()->with('error', 'Current password does not match.');
        }

        if(Hash::check($request->password, $user->password)) {
            return back()->with('error', 'New password cannot be the same as old password.');
        }

        $user->update([
            'password' => Hash::make($request->password)
        ]);
        
        Auth::login($user);
        return redirect(route('profile'))->with('success', 'Login Details updated.');
    }

    public function updateProfilePhoto(Request $request) {
        $request->validate([
            'profile_photo' => 'required|max:4096|mimes:png,jpg,jpeg,'
        ]);

        $user = User::find(auth()->id());
        if($request->hasFile('profile_photo')) {
            $profile_photo_path = '/storage/'.$request->file('profile_photo')->store('profile_images', 'public');

            $user->update([
                'profile_photo_path' => $profile_photo_path
            ]);
        }

        return redirect(route('profile'))->with('success', 'Login Details updated.');
    }
}
